<?php

/**
 * This is the model class for table "lookup_department".
 *
 * The followings are the available columns in table 'lookup_department':
 * @property integer $department_id
 * @property integer $parent_dept_id
 * @property integer $main_dept_id
 * @property string $department_code
 * @property string $department_name
 * @property string $is_active
 * @property string $added_on
 *
 * The followings are the available model relations:
 * @property CutoffOverallGrades[] $cutoffOverallGrades
 * @property CutoffOverallMarks[] $cutoffOverallMarks
 * @property CutoffSubjectGrades[] $cutoffSubjectGrades
 * @property CutoffSubjectMarks[] $cutoffSubjectMarks
 * @property DiSearchDept[] $diSearchDepts
 * @property DiSearchDept[] $diSearchDepts1
 * @property SessionAdvisorDeptCountry[] $sessionAdvisorDeptCountries
 * @property WorkshopGroupDepartment[] $workshopGroupDepartments
 */
class LookupDepartment extends CActiveRecord
{
	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return LookupDepartment the static model class
	 */
    public $department_ref_id;
    public $seo_department_name;

	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'lookup_department';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('department_code, department_name', 'required'),
			array('parent_dept_id, main_dept_id', 'numerical', 'integerOnly'=>true),
			array('department_code', 'length', 'max'=>10),
			array('department_name', 'length', 'max'=>255),
			array('is_active', 'length', 'max'=>1),
			array('added_on', 'safe'),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('department_id, parent_dept_id, main_dept_id, department_code, department_name, is_active, added_on', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'cutoffOverallGrades' => array(self::HAS_MANY, 'CutoffOverallGrades', 'department_ref_id'),
			'cutoffOverallMarks' => array(self::HAS_MANY, 'CutoffOverallMarks', 'department_ref_id'),
			'cutoffSubjectGrades' => array(self::HAS_MANY, 'CutoffSubjectGrades', 'department_ref_id'),
			'cutoffSubjectMarks' => array(self::HAS_MANY, 'CutoffSubjectMarks', 'department_ref_id'),
			'diSearchDepts' => array(self::HAS_MANY, 'DiSearchDept', 'main_dept_ref_id'),
			'diSearchDepts1' => array(self::HAS_MANY, 'DiSearchDept', 'subdept_ref_id'),
			'sessionAdvisorDeptCountries' => array(self::HAS_MANY, 'SessionAdvisorDeptCountry', 'department_ref_id'),
			'workshopGroupDepartments' => array(self::HAS_MANY, 'WorkshopGroupDepartment', 'department_ref_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'department_id' => 'Department',
			'parent_dept_id' => 'Parent Dept',
			'main_dept_id' => 'Main Dept',
			'department_code' => 'Department Code',
			'department_name' => 'Department Name',
			'is_active' => 'Is Active',
			'added_on' => 'Added On',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('department_id',$this->department_id);
		$criteria->compare('parent_dept_id',$this->parent_dept_id);
		$criteria->compare('main_dept_id',$this->main_dept_id);
		$criteria->compare('department_code',$this->department_code,true);
		$criteria->compare('department_name',$this->department_name,true);
		$criteria->compare('is_active',$this->is_active,true);
		$criteria->compare('added_on',$this->added_on,true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}
	
	/**
	 * Returns the array of objects to be used for suggestion list
	 * @param string $searchTerm
	 * @param int $limit, Optional, default 10
	 * @return array
	 */
	public static function getSuggestionList($searchTerm, $limit = 10){
	    $criteria = new CDbCriteria;
	    $criteria->compare('department_name', $searchTerm, true);
	    $criteria->compare('department_code', $searchTerm, true, 'OR');
	    $criteria->compare('is_active', 'Y');
	    $criteria->limit = $limit;
	    
	    return self::model()->findAll($criteria);
	}
	
    public static function getMainDepartmentList() {
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->select = array("department_id" , "department_name");
        $cdbCriteria->addCondition("main_dept_id = 0", "AND");
        $cdbCriteria->addCondition("is_active = 'Y'", "AND");
        $cdbCriteria->order = "department_name";
        
        $modelMainDepts = self::model()->findAll($cdbCriteria);
        $mainDepts = array();
        foreach ($modelMainDepts as $mainDeptsVal) {
            $mainDepts[$mainDeptsVal->attributes['department_id']] = $mainDeptsVal->attributes['department_name'];
        }
        return $mainDepts;
    }

    public static function getSubDepartmentsList($departmentId, $others = true,$limit = false){
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->addCondition("main_dept_id = '$departmentId'", "AND");
        $cdbCriteria->addCondition("is_active = 'Y'", "AND");
        if ($others == false) {
            $cdbCriteria->addCondition(" department_name NOT LIKE  '%others'", "AND");
        }
        if($limit)
        {
           $cdbCriteria->limit = 3;
        }
        $cdbCriteria->order = "department_name ASC";
        $result = self::model()->findAll($cdbCriteria);
        return $result;
    }

    public static function getDepartmentDetailsById($departmentId, $displayFields=array()){
        $cdbCriteria = new CDbCriteria();
        if(count($displayFields) > 0){
            $cdbCriteria->select = implode(', ', $displayFields);
        }
        $cdbCriteria->addCondition("department_id = '$departmentId'", "AND");
        $result = self::model()->find($cdbCriteria);
        return $result;
    }

    public static function  getSubDepartmentsPopularCourses($departmentId, $departmentName = ''){

        if($departmentName == '') {
            $departmentDetails = LookupDepartment::getDepartmentDetailsById($departmentId, array('department_name'));
            $departmentName = $departmentDetails->attributes['department_name'];
        }
         
        $popularCoursesArray = array();
        $popularCoursesArray['courses']['title'] = 'Popular courses in '.$departmentName;
        $popularCoursesList = LookupPopularCourses::getPopularCoursesByDepartment($departmentId);
        if(isset($popularCoursesList)){
            foreach($popularCoursesList as $popularCourse){
                //print_r($popularCourse->attributes);
                //echo "<Br>####<br>";
                $popularCoursesArray['courses']['fullName'][$popularCourse->attributes['popular_courses_id']] = $popularCourse->attributes['course_name'];
            }
        }

        $subDepartmentsArray = array();
        $subDepartmentsArray['departments']['title'] = 'Other courses in '.$departmentName;
        $subDepartmentsList = LookupDepartment::getSubDepartmentsList($departmentId);
        if(isset($subDepartmentsList)){
            foreach($subDepartmentsList as $subDepartment){
                $subDepartmentsArray['departments']['fullName'][$subDepartment->attributes['department_id']] = $subDepartment->attributes['department_name'];
            }
        }
        $returnArray = array_merge($popularCoursesArray, $subDepartmentsArray);
        return $returnArray;
    }

    public static function updateUserPreferedDepartments(){

    }

    public static function getAllSubDepartments(){
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->addCondition("t.main_dept_id > 0", "AND");
        $cdbCriteria->addCondition("t.is_active = 'Y'", "AND");
        $cdbCriteria->order = "t.department_id";
        $result = self::model()->findAll($cdbCriteria);
        return $result;
    }
    
  public static function subDepartmentName($subDeptId){
    $criteria = new CDbCriteria;
    $criteria->compare('department_id', $subDeptId);    
    $res = self::model()->find($criteria);           
    return $res['department_name'];
  }

    public static function getMainDepartmentListByParams($searchParams = array()) {
        $cdbCriteria = new CDbCriteria();
        $cdbCriteria->select = array("department_id" , "department_name");
        $cdbCriteria->addCondition("main_dept_id = 0", "AND");
        $cdbCriteria->addCondition("is_active = 'Y'", "AND");
        $cdbCriteria->order = "department_name";
        if(count($searchParams) > 0 && !empty($searchParams)){
            foreach($searchParams as $fieldName => $fieldValue){
                $cdbCriteria->compare($fieldName, $fieldValue);
            }
        }
        $modelMainDepts = self::model()->findAll($cdbCriteria);
        $mainDepts = array();
        foreach ($modelMainDepts as $mainDeptsVal) {
            $mainDepts[$mainDeptsVal->attributes['department_id']] = $mainDeptsVal->attributes['department_name'];
        }
        return $mainDepts;
    }

    public static function getAllSubDepartmentsList($departmentId){
        $sql = "SELECT department_ref_id,course_name FROM `lookup_popular_courses`
                WHERE main_dept_ref_id = $departmentId
                UNION
                SELECT department_id,department_name FROM lookup_department
                WHERE main_dept_id = $departmentId
                ORDER BY RAND()
                LIMIT 5
        ";
        $result = Yii::app()->db->createCommand($sql)->setFetchMode(PDO::FETCH_OBJ)->queryAll();
        return $result;
    }
    
    public static function _list(){
        $d_results = Yii::app()->db->createCommand()
                ->select(sprintf('LookupDepartment.department_id, LookupDepartment.department_name, LookupDepartment.dept_slug_name, LookupDepartment.dept_alias'))
                ->from(LookupDepartment::model()->tableSchema->name.' LookupDepartment')
                ->where('LookupDepartment.is_active = "Y" AND main_dept_id != 0')
                ->queryAll();
        $data = array();
        
        foreach($d_results as $d){
            $data[$d['department_id']]['id'] = $d['department_id'];
            $data[$d['department_id']]['title'] = $d['department_name'];
            $data[$d['department_id']]['slug'] = $d['dept_slug_name'];
            $data[$d['department_id']]['dept_alias'] = $d['dept_alias'];
        }
        return $data;
        
    }
    
    public static function get_main_department_list($key_column = 'department_id', $value_column = 'department_name'){
        $d_results = Yii::app()->db->createCommand()
                ->select(sprintf('LookupDepartment.%s, LookupDepartment.%s', $key_column, $value_column))
                ->from(LookupDepartment::model()->tableSchema->name.' LookupDepartment')
                ->where('LookupDepartment.is_active = "Y" AND main_dept_id = 0')
                ->queryAll();
        
        return CHtml::listData($d_results, $key_column, $value_column);
    }
    
    
    public static function get_department_list(){
        $d_results = Yii::app()->db->createCommand()
                ->select('LookupDepartment.department_id, LookupDepartment.department_name, LookupDepartment.dept_slug_name, LookupDepartment.dept_alias')
                ->from(LookupDepartment::model()->tableSchema->name.' LookupDepartment')
                ->where('LookupDepartment.is_active = "Y" AND main_dept_id = 0')
                ->order('LookupDepartment.department_name')
                ->queryAll();
        $data = array();
        
        foreach($d_results as $d){
            $data[$d['dept_slug_name']]['id'] = $d['department_id'];
            $data[$d['dept_slug_name']]['title'] = $d['department_name'];
            $data[$d['dept_slug_name']]['slug'] = $d['dept_slug_name'];
            $data[$d['dept_slug_name']]['dept_alias'] = $d['dept_alias'];
        }
        return $data;
    }
    
}