<?php
Yii::import('application.modules.sms.models.base.BaseStudentIdealListCollege');

/**
 * Model class for StudentIdealListCollege
 *
 *
 * @author Shiva <shiva.dharana@univariety.com>
 * @version 1.0
 * @package sms.models
 *
 */
class StudentIdealListCollege extends BaseStudentIdealListCollege{

    /**
     * Statuses of this model
     * @var const
     */
    const STATUS_PENDING = 'Pending';
    const STATUS_REVIEWED = 'Reviewed';
    const STATUS_ACCEPTED = 'Accept';
    const STATUS_REJECTED = 'Reject';

    const SUGGEST_TYPE_ADVISOR = 'Advisor';
    const SUGGEST_TYPE_STUDENT = 'Student';
    const SUGGEST_TYPE_SPOC = 'Spoc';
    
    const APPLYING = 1;
    const NOT_APPLYING = 0;


    public static function model($className=__CLASS__){
        return parent::model($className);
    }
    
	/**
     * @return array customized attribute labels (name=>label)
     */
    public function attributeLabels()
    {
        return array(
			'student_ideal_list_college_id' => 'Student Ideal List College',
			'student_ideal_list_ref_id' => 'Student Ideal List',
			'uni_ref_id' => 'University',
			'suggest_type' => 'Suggest Type',
			'suggested_by' => 'Suggested By',
			'comment' => 'Comment',
			'comment_by' => 'Comment By',
			'status' => 'Status',
			'conclude_date' => 'Conclude Date',
			'review_date' => 'Review Date',
			'added_on' => 'Added On',
            'is_applying' => 'Applying',
            'preference_order' => 'Preference Order',
        );
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules(){
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
            
            array('student_ideal_list_ref_id, uni_ref_id, suggested_by', 'required'),
            array('uni_ref_id', 'numerical', 'integerOnly'=>true),
            array('student_ideal_list_ref_id, suggested_by, comment_by', 'length', 'max'=>10),
            array('suggest_type, status', 'length', 'max'=>10),
            array('comment, conclude_date, review_date, added_on, preference_order', 'safe'),
            
            array('*', 'isDuplicate', 'on' => 'insert'),
            array('*', 'checkMax', 'on' => 'insert'),
            array('student_ideal_list_college_id, comment', 'required', 'on' => 'review'),
            
            // The following rule is used by search().
            // Please remove those attributes that should not be searched.
            array('student_ideal_list_college_id, student_ideal_list_ref_id, uni_ref_id, suggest_type, suggested_by, comment, comment_by, status, conclude_date, review_date, added_on', 'safe', 'on'=>'search'),
        );
    }

    /**
     * @return array relational rules.
     */
    public function relations(){
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        return array(
			'studentIdealList' => array(self::BELONGS_TO, 'StudentIdealList', 'student_ideal_list_ref_id'),
			'university' => array(self::BELONGS_TO, 'TblUniversityMaster', 'uni_ref_id'),
			'suggestedBy' => array(self::BELONGS_TO, 'LoginMaster', 'suggested_by'),
			'commentBy' => array(self::BELONGS_TO, 'LoginMaster', 'comment_by'),
        );


    }

    /**
     * Adds a university to the list.
     * @return bool
     */
    public function addUniversity($studentIdealList){
        // Change the status of the university based on who added it.
        // All student added universities should be reviewed by advisor.
        switch($this->suggest_type){
            case self::SUGGEST_TYPE_STUDENT:
                $status = self::STATUS_PENDING;
                break;
            case self::SUGGEST_TYPE_SPOC:
            case self::SUGGEST_TYPE_ADVISOR:
                $status = self::STATUS_REVIEWED;
                break;
        }


        // ideal and safe list are added by advisor, and can't be edited. so status will be accepted
        if($studentIdealList->type == StudentIdealList::TYPE_IDEAL || $studentIdealList->type == StudentIdealList::TYPE_SAFE){
            $status = self::STATUS_ACCEPTED;
        }

        $this->student_ideal_list_ref_id = $studentIdealList->student_ideal_list_id;
        $this->suggested_by = Yii::app()->user->id;
        $this->status = $status;
        return $this->save();
    }

    public function isDuplicate(){
        if(empty($this->uni_ref_id)){
            return true;
        }
        
        /**
         * check for duplicate records across the lists.
         */
        $listIds = array();
//        $departments = $this->studentIdealList->student->studentPreferredDepartments;
        $listTypes = StudentIdealList::getTypeLabels();

        /*foreach($departments as $department){*/
            foreach($listTypes as $listType => $label){
                $studentIdealList = StudentIdealList::getRecord($this->studentIdealList->user_ref_id, $this->studentIdealList->department_ref_id, $listType, false);
                $listIds[] = $studentIdealList->student_ideal_list_id;
            }
     /*   }*/

        
        $criteria = new CDbCriteria();
        $criteria->compare('student_ideal_list_ref_id', $listIds);
        $criteria->compare('uni_ref_id', $this->uni_ref_id);
        
        $record = self::model()->find($criteria);
        //$record = null;
        if(is_null($record)){
            return true;
        }
        
        //var_dump($record->attributes);
        if($this->suggest_type == self::SUGGEST_TYPE_STUDENT && ($record->suggest_type == self::SUGGEST_TYPE_ADVISOR || $record->suggest_type == self::SUGGEST_TYPE_SPOC)){
            $record->delete();
            return true;
        }
        
        $this->addError('*', 'You Have Already Added This University.');
        return false;
        
    }

    public function checkMax(){
        $list = StudentIdealList::model()->findByPk($this->student_ideal_list_ref_id);
        $cnt = 0;
        switch($this->suggest_type){
            case self::SUGGEST_TYPE_ADVISOR: $cnt = $list->advisorAddedCount; break;
            case self::SUGGEST_TYPE_SPOC: $cnt = $list->spocAddedCount; break;
            case self::SUGGEST_TYPE_STUDENT: $cnt = $list->studentAddedCount; break;
        }
        if($cnt >= 8){
            $this->addError('*', 'You cannot add more than 8 universities');
            return false;
        }
        return true;
    }

    protected function beforeSave(){
        if(parent::beforeSave()){
            if($this->isNewRecord){
                $this->added_on = date('Y-m-d H:i:s');
            }else{
                $this->conclude_date = date('Y-m-d H:i:s');
            }
            return true;
        }
        return false;
    }

    protected function afterSave(){
        parent::afterSave();
        
        // Add to audit log.
        $log = new StudentIdealListCollegeLog;
        $log->addAuditLog($this);
    }
    
    protected function afterDelete(){
        parent::afterDelete();
        
        // Add to audit log.
        $log = new StudentIdealListCollegeLog;
        $log->addAuditLog($this, true);
    }
    
    public static function getApplyingListCount($student, $departmentId){
        
    	/**
         * check for duplicate records across the lists.
         */
        $listIds = array();
        $department = LookupDepartment::model()->findByPk($departmentId);
        
        $listTypes = StudentIdealList::getTypeLabels();
            
        foreach($listTypes as $listType => $label){
            $studentIdealList = StudentIdealList::getRecord($student->user_id, $department->department_id, $listType);
            $listIds[] = $studentIdealList->student_ideal_list_id;
        }
        
        $criteria = new CDbCriteria();
		$criteria->compare('student_ideal_list_ref_id', $listIds);
		$criteria->compare('is_applying', 1);
		
		return self::model()->count($criteria);
        
    }
    
    public static function getStudentIdealCollegeListForCompareCollege($userId){

        $data = Yii::app()->db->createCommand()
            ->select('uni_ref_id,uni_name')
            ->from('student_ideal_list AS sil')
            ->join('student_preferred_department AS spd', 'sil.user_ref_id = spd.user_ref_id AND spd.main_dept_ref_id = sil.department_ref_id')
            ->join('student_ideal_list_college AS silc', 'silc.student_ideal_list_ref_id = sil.student_ideal_list_id')
            ->join('tbl_university_master AS tum', 'silc.uni_ref_id = tum.uni_id')
            ->where('sil.user_ref_id=:id AND sil.type="Aspirational" AND silc.suggest_type="Student"', array(':id' => $userId))
            ->group('uni_id')
            ->queryAll();
        return $data;
        
    }
}